/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.assertions;

import static org.apache.juneau.assertions.AssertionPredicates.ne;
import static org.apache.juneau.assertions.Assertions.*;
import static org.apache.juneau.commons.utils.CollectionUtils.*;
import static org.apache.juneau.commons.utils.Utils.*;
import static org.junit.jupiter.api.Assertions.*;

import java.util.*;

import org.apache.juneau.*;
import org.apache.juneau.commons.utils.*;
import org.apache.juneau.json.*;
import org.junit.jupiter.api.*;

@Deprecated
class BeanListAssertion_Test extends TestBase {

	//------------------------------------------------------------------------------------------------------------------
	// Helpers
	//------------------------------------------------------------------------------------------------------------------

	private static BeanListAssertion<A> test(List<A> value) {
		return assertBeanList(value).setSilent();
	}
	private static final A A1 = A.of(1,2), A1a = A.of(1,2), A2 = A.of(3,4), A3 = A.of(5,6);

	public static class A implements Comparable<A> {
		public Integer a, b;
		public A() {}
		public A(Integer a, Integer b) { this.a = a; this.b = b; }
		public static A of(Integer a, Integer b) { return new A(a, b); }
		@Override public String toString() { return "(a="+a+",b="+b+")"; }
		@Override public boolean equals(Object o) { return eq(this, (A)o, (x,y)->eq(x.a,y.a) && eq(x.b,y.b)); }
		@Override public int compareTo(A o) { return a-o.a; }
	}

	//-----------------------------------------------------------------------------------------------------------------
	// Basic tests
	//-----------------------------------------------------------------------------------------------------------------

	@Test void a01_msg() {
		assertThrows(BasicAssertionError.class, ()->test(null).setMsg("Foo {0}", 1).isExists(), "Foo 1");
		assertThrows(RuntimeException.class, ()->test(null).setMsg("Foo {0}", 1).setThrowable(RuntimeException.class).isExists(), "Foo 1");
	}

	@Test void a02_stdout() {
		test(null).setStdOut();
	}

	//-----------------------------------------------------------------------------------------------------------------
	// Transform tests
	//-----------------------------------------------------------------------------------------------------------------

	@Test void ba01a_asString() {
		var x = l(A1);
		var nil = (List<A>)null;
		test(x).asString().is("[(a=1,b=2)]");
		test(nil).asString().isNull();
	}

	@Test void ba01b_asString_wSerializer() {
		var x = l(A1);
		var nil = (List<A>)null;
		var s = Json5Serializer.DEFAULT;
		test(x).asString(s).is("[{a:1,b:2}]");
		test(nil).asString(s).is("null");
	}

	@Test void ba01c_asString_wPredicate() {
		var x1 = l(A1);
		test(x1).asString(x -> "foo").is("foo");
	}

	@Test void ba02_asJson() {
		var x = l(A1);
		var nil = (List<A>)null;
		test(x).asJson().is("[{a:1,b:2}]");
		test(nil).asJson().is("null");
	}

	@Test void ba03_asJsonSorted() {
		var x1 = l(A2,A1);
		var nil = (List<A>)null;
		test(x1).asJsonSorted().is("[{a:1,b:2},{a:3,b:4}]");
		test(nil).asJsonSorted().is("null");
	}

	@Test void ba04_apply() {
		var x1 = l(A1);
		var x2 = l(A2);
		test(x1).asTransformed(x -> x2).is(x2);
	}

	@Test void bb01_asStrings() {
		var x1 = l(A1);
		var nil = (List<A>)null;
		test(x1).asStrings().asJoin().is("(a=1,b=2)");
		test(nil).asStrings().isNull();
	}

	@Test void bb02_size() {
		var x1 = l(A1);
		var nil = (List<A>)null;
		test(x1).asSize().is(1);
		test(nil).asSize().isNull();
	}

	@Test void bc01_apply2() {
		var x1 = l(A1);
		var x2 = l(A2);
		test(x1).asApplied2(x -> x2).is(x2);
	}

	@Test void bc02_item() {
		var x = l(A1);
		var nil = (List<A>)null;
		test(x).asItem(0).isNotNull();
		test(x).asItem(1).isNull();
		test(x).asItem(-1).isNull();
		test(nil).asItem(0).isNull();
	}

	@Test void bc03a_sorted() {
		var x = l(A2,A1);
		var nil = (List<A>)null;
		test(x).asSorted().isString("[(a=1,b=2),(a=3,b=4)]");
		test(nil).asSorted().isNull();
	}

	@Test void bc03b_sorted_wComparator() {
		var x = l(A2,A1);
		var nil = (List<A>)null;
		test(x).asSorted(null).isString("[(a=1,b=2),(a=3,b=4)]");
		test(nil).asSorted(null).isNull();
	}

	@Test void bd01_extract() {
		var x = l(A1,A2);
		test(x)
			.asPropertyMaps("a").asJson().is("[{a:1},{a:3}]")
			.asPropertyMaps("a,b").asJson().is("[{a:1,b:2},{a:3,b:4}]")
			.asPropertyMaps("a","b").asJson().is("[{a:1,b:2},{a:3,b:4}]")
			.asPropertyMaps("bad").asJson().is("[{},{}]")
			.asPropertyMaps((String)null).asJson().is("[{},{}]");
	}

	@Test void bd02_property() {
		var x = l(A1,A2);
		test(x)
			.asProperty("a").asJson().is("[1,3]")
			.asProperty("bad").asJson().is("[null,null]")
			.asProperty(null).asJson().is("[null,null]");
	}

	//-----------------------------------------------------------------------------------------------------------------
	// Test tests
	//-----------------------------------------------------------------------------------------------------------------

	@Test void ca01_exists() {
		var x = CollectionUtils.<A>l();
		var nil = (List<A>)null;
		test(x).isExists().isExists();
		assertThrows(BasicAssertionError.class, ()->test(nil).isExists(), "Value was null.");
	}

	@Test void ca02_isNull() {
		var x = CollectionUtils.<A>l();
		var nil = (List<A>)null;
		test(nil).isNull();
		assertThrows(BasicAssertionError.class, ()->test(x).isNull(), "Value was not null.");
	}

	@Test void ca03_isNotNull() {
		var x = CollectionUtils.<A>l();
		var nil = (List<A>)null;
		test(x).isNotNull();
		assertThrows(BasicAssertionError.class, ()->test(nil).isNotNull(), "Value was null.");
	}

	@Test void ca04a_is_T() {
		var x1 = l(A1,A2);
		var x1a = l(A1a,A2);
		var x2 = l(A1,A3);
		var nil = (List<A>)null;
		test(x1).is(x1);
		test(x1).is(x1a);
		test(nil).is(nil);
		assertThrown(()->test(x1).is(x2)).asMessage().asOneLine().is("Unexpected value.  Expect='[(a=1,b=2),(a=5,b=6)]'.  Actual='[(a=1,b=2),(a=3,b=4)]'.");
		assertThrown(()->test(x1).is(nil)).asMessage().asOneLine().is("Unexpected value.  Expect='null'.  Actual='[(a=1,b=2),(a=3,b=4)]'.");
		assertThrown(()->test(nil).is(x2)).asMessage().asOneLine().is("Unexpected value.  Expect='[(a=1,b=2),(a=5,b=6)]'.  Actual='null'.");
	}

	@Test void ca04b_is_predicate() {
		var x1 = l(A1,A2);
		test(x1).is(x->x.size()==2);
		assertThrown(()->test(x1).is(x->x.size()==3)).asMessage().asOneLine().is("Unexpected value: '[(a=1,b=2),(a=3,b=4)]'.");
		assertThrown(()->test(x1).is(ne(x1))).asMessage().asOneLine().is("Value unexpectedly matched.  Value='[(a=1,b=2),(a=3,b=4)]'.");
	}

	@Test void ca05_isNot() {
		var x1 = l(A1,A2);
		var x1a = l(A1a,A2);
		var x2 = l(A1,A3);
		var nil = (List<A>)null;
		test(x1).isNot(x2);
		test(x1).isNot(nil);
		test(nil).isNot(x1);
		assertThrown(()->test(x1).isNot(x1a)).asMessage().asOneLine().is("Unexpected value.  Did not expect='[(a=1,b=2),(a=3,b=4)]'.  Actual='[(a=1,b=2),(a=3,b=4)]'.");
		assertThrown(()->test(nil).isNot(nil)).asMessage().asOneLine().is("Unexpected value.  Did not expect='null'.  Actual='null'.");
	}

	@Test void ca06_isAny() {
		var x1 = l(A1,A2);
		var x1a = l(A1a,A2);
		var x2 = l(A1,A3);
		var nil = (List<A>)null;
		test(x1).isAny(x1a, x2);
		assertThrown(()->test(x1).isAny(x2)).asMessage().asOneLine().is("Expected value not found.  Expect='[[(a=1,b=2),(a=5,b=6)]]'.  Actual='[(a=1,b=2),(a=3,b=4)]'.");
		assertThrown(()->test(x1).isAny()).asMessage().asOneLine().is("Expected value not found.  Expect='[]'.  Actual='[(a=1,b=2),(a=3,b=4)]'.");
		assertThrown(()->test(nil).isAny(x2)).asMessage().asOneLine().is("Expected value not found.  Expect='[[(a=1,b=2),(a=5,b=6)]]'.  Actual='null'.");
	}

	@Test void ca07_isNotAny() {
		var x1 = l(A1,A2);
		var x1a = l(A1a,A2);
		var x2 = l(A1,A3);
		var nil = (List<A>)null;
		test(x1).isNotAny(x2);
		test(x1).isNotAny();
		test(nil).isNotAny(x2);
		assertThrown(()->test(x1).isNotAny(x1a)).asMessage().asOneLine().is("Unexpected value found.  Unexpected='[(a=1,b=2),(a=3,b=4)]'.  Actual='[(a=1,b=2),(a=3,b=4)]'.");
		assertThrown(()->test(nil).isNotAny(nil)).asMessage().asOneLine().is("Unexpected value found.  Unexpected='null'.  Actual='null'.");
	}

	@Test void ca08_isSame() {
		var x1 = l(A1,A2);
		var x1a = l(A1a,A2);
		var nil = (List<A>)null;
		test(x1).isSame(x1);
		test(nil).isSame(nil);
		assertThrown(()->test(x1).isSame(x1a)).asMessage().asOneLine().isMatches("Not the same value.  Expect='[(a=1,b=2),(a=3,b=4)](*)'.  Actual='[(a=1,b=2),(a=3,b=4)](*)'.");
		assertThrown(()->test(nil).isSame(x1a)).asMessage().asOneLine().isMatches("Not the same value.  Expect='[(a=1,b=2),(a=3,b=4)](*)'.  Actual='null(null)'.");
		assertThrown(()->test(x1).isSame(nil)).asMessage().asOneLine().isMatches("Not the same value.  Expect='null(null)'.  Actual='[(a=1,b=2),(a=3,b=4)](*)'.");
	}

	@Test void ca09_isSameJsonAs() {
		var x1 = l(A1,A2);
		var x1a = l(A1a,A2);
		var x2 = l(A1,A3);
		var nil = (List<A>)null;
		test(x1).isSameJsonAs(x1a);
		test(nil).isSameJsonAs(nil);
		assertThrown(()->test(x1a).isSameJsonAs(x2)).asMessage().asOneLine().is("Unexpected comparison.  Expect='[{a:1,b:2},{a:5,b:6}]'.  Actual='[{a:1,b:2},{a:3,b:4}]'.");
		assertThrown(()->test(nil).isSameJsonAs(x2)).asMessage().asOneLine().is("Unexpected comparison.  Expect='[{a:1,b:2},{a:5,b:6}]'.  Actual='null'.");
		assertThrown(()->test(x1).isSameJsonAs(nil)).asMessage().asOneLine().is("Unexpected comparison.  Expect='null'.  Actual='[{a:1,b:2},{a:3,b:4}]'.");
	}

	@Test void ca10_isSameSortedJsonAs() {
		var x1 = l(A1,A2);
		var x1a = l(A1a,A2);
		var x2 = l(A1,A3);
		var nil = (List<A>)null;
		test(x1).isSameSortedJsonAs(x1a);
		test(nil).isSameSortedJsonAs(nil);
		assertThrown(()->test(x1a).isSameSortedJsonAs(x2)).asMessage().asOneLine().is("Unexpected comparison.  Expect='[{a:1,b:2},{a:5,b:6}]'.  Actual='[{a:1,b:2},{a:3,b:4}]'.");
		assertThrown(()->test(nil).isSameSortedJsonAs(x2)).asMessage().asOneLine().is("Unexpected comparison.  Expect='[{a:1,b:2},{a:5,b:6}]'.  Actual='null'.");
		assertThrown(()->test(x1).isSameSortedJsonAs(nil)).asMessage().asOneLine().is("Unexpected comparison.  Expect='null'.  Actual='[{a:1,b:2},{a:3,b:4}]'.");
	}

	@Test void ca11_isSameSerializedAs() {
		var x1 = l(A1,A2);
		var x1a = l(A1a,A2);
		var x2 = l(A1,A3);
		var nil = (List<A>)null;
		var s = Json5Serializer.DEFAULT;
		test(x1).isSameSerializedAs(x1a, s);
		test(nil).isSameSerializedAs(nil, s);
		assertThrown(()->test(x1a).isSameSerializedAs(x2, s)).asMessage().asOneLine().is("Unexpected comparison.  Expect='[{a:1,b:2},{a:5,b:6}]'.  Actual='[{a:1,b:2},{a:3,b:4}]'.");
		assertThrown(()->test(nil).isSameSerializedAs(x2, s)).asMessage().asOneLine().is("Unexpected comparison.  Expect='[{a:1,b:2},{a:5,b:6}]'.  Actual='null'.");
		assertThrown(()->test(x1).isSameSerializedAs(nil, s)).asMessage().asOneLine().is("Unexpected comparison.  Expect='null'.  Actual='[{a:1,b:2},{a:3,b:4}]'.");
	}

	@Test void ca12_isType() {
		var x = l(A1,A2);
		var nil = (List<A>)null;
		test(x).isType(List.class);
		test(x).isType(Object.class);
		assertThrown(()->test(x).isType(String.class)).asMessage().asOneLine().isMatches("Unexpected type.  Expect='java.lang.String'.  Actual='java.util.Array*'.");
		assertThrown(()->test(nil).isType(String.class)).asMessage().asOneLine().is("Value was null.");
		assertThrown(()->test(x).isType(null)).asMessage().asOneLine().is("Argument 'parent' cannot be null.");
	}

	@Test void ca13_isExactType() {
		var x = l(A1,A2);
		var nil = (List<A>)null;
		assertThrown(()->test(x).isExactType(String.class)).asMessage().asOneLine().isMatches("Unexpected type.  Expect='java.lang.String'.  Actual='java.util.Array*'.");
		assertThrown(()->test(nil).isExactType(String.class)).asMessage().asOneLine().is("Value was null.");
		assertThrown(()->test(x).isExactType(null)).asMessage().asOneLine().is("Argument 'parent' cannot be null.");
	}

	@Test void ca14_isString() {
		var x = l(A1,A2);
		var nil = (List<A>)null;
		test(x).isString("[(a=1,b=2),(a=3,b=4)]");
		test(nil).isString(null);
		assertThrown(()->test(x).isString("bad")).asMessage().asOneLine().is("String differed at position 0.  Expect='bad'.  Actual='[(a=1,b=2),(a=3,b=4)]'.");
		assertThrown(()->test(x).isString(null)).asMessage().asOneLine().is("String differed at position 0.  Expect='null'.  Actual='[(a=1,b=2),(a=3,b=4)]'.");
		assertThrown(()->test(nil).isString("bad")).asMessage().asOneLine().is("String differed at position 0.  Expect='bad'.  Actual='null'.");
	}

	@Test void ca15_isJson() {
		var x = l(A1,A2);
		var nil = (List<A>)null;
		test(x).isJson("[{a:1,b:2},{a:3,b:4}]");
		test(nil).isJson("null");
		assertThrown(()->test(x).isJson("bad")).asMessage().asOneLine().is("String differed at position 0.  Expect='bad'.  Actual='[{a:1,b:2},{a:3,b:4}]'.");
		assertThrown(()->test(x).isJson(null)).asMessage().asOneLine().is("String differed at position 0.  Expect='null'.  Actual='[{a:1,b:2},{a:3,b:4}]'.");
		assertThrown(()->test(nil).isJson("bad")).asMessage().asOneLine().is("String differed at position 0.  Expect='bad'.  Actual='null'.");
	}

	@Test void cb01_isEmpty() {
		var x1 = CollectionUtils.<A>l();
		var x2 = l(A1);
		var nil = (List<A>)null;
		test(x1).isEmpty();
		assertThrows(BasicAssertionError.class, ()->test(nil).isEmpty(), "Value was null.");
		assertThrows(BasicAssertionError.class, ()->test(x2).isEmpty(), "Collection was not empty.");
	}

	@Test void cb02_isNotEmpty() {
		var x1 = CollectionUtils.<A>l();
		var x2 = l(A1);
		var nil = (List<A>)null;
		test(x2).isNotEmpty();
		assertThrows(BasicAssertionError.class, ()->test(nil).isNotEmpty(), "Value was null.");
		assertThrows(BasicAssertionError.class, ()->test(x1).isNotEmpty(), "Collection was empty.");
	}

	@Test void cb03_contains() {
		var x = l(A1);
		var nil = (List<A>)null;
		test(x).isContains(A1);
		assertThrown(()->test(x).isContains(A2)).asMessage().asOneLine().is("Collection did not contain expected value.  Expect='(a=3,b=4)'.  Value='[(a=1,b=2)]'.");
		assertThrows(BasicAssertionError.class, ()->test(nil).isContains(A2), "Value was null.");
	}

	@Test void cb04_doesNotContain() {
		var x = l(A1);
		var nil = (List<A>)null;
		test(x).isNotContains(A2);
		assertThrown(()->test(x).isNotContains(A1)).asMessage().asOneLine().is("Collection contained unexpected value.  Unexpected='(a=1,b=2)'.  Value='[(a=1,b=2)]'.");
		assertThrows(BasicAssertionError.class, ()->test(nil).isNotContains(A2), "Value was null.");
	}

	@Test void cb05_any() {
		var x1 = l(A1);
		var nil = (List<A>)null;
		test(x1).isAny(x->x.equals(A1));
		assertThrown(()->test(x1).isAny(x->x.equals(A2))).asMessage().asOneLine().is("Collection did not contain tested value.  Value='[(a=1,b=2)]'.");
		assertThrows(BasicAssertionError.class, ()->test(nil).isAny(x->x.equals(A2)), "Value was null.");
	}

	@Test void cb06_all() {
		var x1 = l(A1);
		var nil = (List<A>)null;
		test(x1).isAll(x->x!=null);
		assertThrown(()->test(x1).isAll(x->x.equals(A2))).asMessage().asOneLine().is("Collection did not contain tested value.  Value='[(a=1,b=2)]'.");
		assertThrows(BasicAssertionError.class, ()->test(nil).isAll(x->x.equals(A2)), "Value was null.");
	}

	@Test void cb07_isSize() {
		var x = l(A1);
		var nil = (List<A>)null;
		test(x).isSize(1);
		assertThrown(()->test(x).isSize(0)).asMessage().asOneLine().is("Collection did not have the expected size.  Expect=0.  Actual=1.");
		assertThrows(BasicAssertionError.class, ()->test(nil).isSize(0), "Value was null.");
	}

	@Test void cc01_has() {
		var x = l(A1,A2);
		var nil = (List<A>)null;
		test(x).isHas(A1,A2);
		assertThrown(()->test(x).isHas(A1)).asMessage().asOneLine().is("Collection did not have the expected size.  Expect=1.  Actual=2.");
		assertThrown(()->test(x).isHas(A1,A3)).asMessage().asOneLine().is("List did not contain expected value at index 1.  Value did not match expected.  Expect='(a=5,b=6)'.  Actual='(a=3,b=4)'.");
		assertThrows(BasicAssertionError.class, ()->test(nil).isHas(A1,A3), "Value was null.");
	}

	@Test void cc02_each() {
		var x1 = l(A1,A2);
		var nil = (List<A>)null;
		test(x1).isEach(x->x!=null,x->x!=null);
		assertThrown(()->test(x1).isEach(x->x==null)).asMessage().asOneLine().is("Collection did not have the expected size.  Expect=1.  Actual=2.");
		assertThrown(()->test(x1).isEach(x->x==null,x->x==null)).asMessage().asOneLine().is("List did not contain expected value at index 0.  Unexpected value: '(a=1,b=2)'.");
		assertThrows(BasicAssertionError.class, ()->test(nil).isEach(x->x==null), "Value was null.");
	}
}